/*
 * Simon's Sequencer V3
 * 
 * 
 * Some conventions:
 *  each function area (systems) have their own ino page and/or class
 *  SSPanel is an Arduino library because it can be used for other projects and its implimentation is independent of sequencer functions (eg. it does know what a play button is, it is just the green button)
 *  SSClock, SSNoteData and SSTrigData are classes because I wanted to have encapsilation, not sure why, but in general I see Classes as a pain if you are not going to have multiple instances, so... 
 *  ALL the other systems are puedo classes. They have member variables at the top and "public" and "private" functions although this is not really enforced (because I'm lazy)
 *  variables starting with underscore are global across the entire project
 *  variables starting with "m" are members to the system
 *  functions starting with the system name (eg ssClockHelper_) are meant to be public; this helps with navigating around the code
 *  functions within a system without the system name are private and should only be called from the system's page
 *  some systems have _Defs.h files because they share definitions across multiple systems (eg  settings structures are used throughout)
 *  
 *  
 */
 
/*
MIT License

Copyright (c) 2023 Simon R. Smith

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/

#define DEBUG_SHOWSTEPS   16 
#define MAXSTEPS          256

#include <SSPanel.h>
#include "SSClock.h"
#include <DueTimer.h>
#include <malloc.h>
#include <stdint.h>
#include <MIDI.h>
#include <SPI.h>
#include <SdFat.h>
#include "SSNoteData.h"
#include "SSTrigData.h"
#include "SSSettings_Defs.h"
#include "SSRobots_Defs.h"
#include "SSDisplayUI_Defs.h"
#include "SSString_Defs.h"
         
#define SD_CS    2  // Chip select line for SD card on Shield

#define FILEBREAK 0x8000
#define EXTCLOCKAVGCOUNTMAX 5

#define DEFAULTMIDITRIGGERCHANNEL 10

#define NOCTRLBTN     (ctrl == NOBUTTON)
#define NOSELBTN      (select == NOBUTTON)
  
#define NUMTRACKS         8     
#define DEFAULT_LENGTH    2      // in bars
#define DEFAULT_TEMPO     120

#define LIVE  1
#define SEQ   2
#define BOTH  3

// panel pages
#define DEFAULT       0
#define BARLENGTH     1
#define NOTEMIDIMAP   2
#define BARJUMP       3
#define PATCHSELECT   4
#define STEPJUMP      5
#define COPYTRACK     6
#define COPYBANK      7
#define MUTEPAGE      8
#define SOLOPAGE      9
#define SETTINGS      10
#define FILEPAGE      11
#define SAVEDMAP      12
#define DMAPSELECT    13
#define ROBOTPAGE1    14
#define ROBOTPAGE2    15
#define ROBOTCOMPLEX  16
#define ROBOTPROB     17
#define ROBOTSCALE    18
#define STEPEDIT      19
#define VMAPHIT       20
#define VMAPACCENT    21
#define STEPRECORD    22
#define ROBOTCHORD    23
#define SETSHORTCUT   24
#define SWINGVALUE    25
#define ROBOTPLEN     26
#define COPYDRUMS     27
#define OPTIONDIALOG  28
#define RESTOREPAGE   29

// midi handlers
#define INVALID   0
#define SEQUENCER 1
#define TRIGGER   2
#define TURING    3
#define INVERVAL  4
#define ARP       5
#define STEPREC   6
#define CHORD     7

// confirm page
#define NOCONFIRM   0
#define CLEARALL    1
#define CLEARBANK   2
#define CLEARDRUMS  3
#define CLEARTRACK  4
#define CLEARNEW    5

// lock pages
#define NOLOCK      0
#define SOLOLOCK    1
#define MUTELOCK    2
#define STEPLOCK    3
#define PATCHLOCK   4

MIDI_CREATE_INSTANCE(HardwareSerial, Serial1, MIDI)  
MIDI_CREATE_INSTANCE(HardwareSerial, Serial2, MIDI2)  

struct KeyInputState{
  byte note;
  byte velocity;
  byte handler;
  byte bank;
  byte track;
  bool recorded;
  byte port;
};

struct SimpleFileSystem{
  uint16_t noteDataSizeBlockA;
  uint16_t noteDataSizeBlockB;
  uint16_t trigDataSize;
  uint16_t muteStateSize;
  uint16_t soloStateSize;
  uint16_t projectSettingsSize;
  uint16_t extension1Size;
  uint16_t extension2Size;
};

SSPanel ssPanel;
SSNoteData ssNoteData;
SSTrigData ssTrigData;

/*
 * 
 *    GLOBAL STATE
 * 
 */
uint32_t _MuteState[4] = {0,0,0,0};
uint32_t _SoloState[4] = {0,0,0,0};
ProjectSettings _projectSettings; 
 
// Sequencer State
bool _metro = true;
byte _currentBank = 0;
byte _nextBank = 0;
byte _currentNoteTrack = 0;
byte _currentTriggerTrack = 0;
bool _inTrigger = false;
bool _showSuperTriggerUI = false;
int  _currentStep = -1;
bool _inStepRec = false;
bool _inRec = false;
bool _inPlay = false;
bool _inErase = false;
bool _inKeyboard = false;
byte _inCountIn = 0;
byte _inFill = 0;
byte _inSlide = 0;
bool _inRatchet = false;
bool _haveDownBeat = false;
bool _recUnQuantized = false;
byte _muteSoloEditBank = _currentBank;

// robot handler
byte _activeRobot = 0xFF;

// Display State
byte _panelPage = DEFAULT;
byte _rtnPage = DEFAULT;
bool _inPopup = false;
byte _confirmPage = NOCONFIRM;
byte _currentPatchBank = 0;
byte _currentMidiPort = 0;        // false = portA 
byte _bankTrackCopyBank = 0;
bool _copyDrumsBank = false;
byte _inLock = 0;
bool _inKnobPush = false;
bool _inShift = false;

bool _sdActive;
File _root;       // SSFileHelper and SSSettings use _root 
SdFat SD;

void setup() {
  Serial.begin(250000);
  Serial.println();  Serial.println();
  Serial.println(F(" DEBUGGING "));
  Serial.println();  Serial.println();

  freeRam(); 
  
  if (SD.begin(SD_CS)) {
    _sdActive = true;                        // active the SD card
  }

  // set PANEL interupt and event handlers
  Timer1.attachInterrupt(panelTimerInterrupt);                                  // used to scan LEDS, drive the buzzer for the metronome and some async UI events like Sliding the LEDS and clearing pop-up messages
  ssPanel.begin();
  attachInterrupt(digitalPinToInterrupt(KNOB_A), panelRotateInterupt, CHANGE);  // knob turn is interupt driven
  attachInterrupt(digitalPinToInterrupt(KNOB_B), panelRotateInterupt, CHANGE); 
  ssPanel.setHandlerSelectButtonEvent(ssUIHandler_handleSelectButton);          // SSPanel throws larious events when buttons are pressed that are then distributed through SSUIHandler
  ssPanel.setHandlerShiftButtonEvent(ssUIHandler_handleShiftButton);
  ssPanel.setHandlerAuxButtonEvent(ssUIHandler_handleAuxButton);
  ssPanel.setHandlerKnobButtonEvent(ssUIHandler_handleKnobButton);
  ssPanel.setHandlerKnobRotateEvent(ssUIHandler_handleKnobRotate);
  ssPanel.setHandlerTrigButtonEvent(ssUIHandler_handleTrigButton);
  ssPanel.setHandlerReturnFromPopup(ssDisplayUI_handleReturnOfPopup);           // this is thrown when the timer above wants to clear a pop-up. I suppose SSDisplay could of had its own timer but I decided to share them
  ssPanel.setHandlerShowMainDialog(ssDisplayUI_handleReturnFromDialog);
  
  // set NOTEDATA and TRIGDATA handlers
  ssNoteData.setHandlerStepPlayback(ssSequencer_handleNoteDataStepPlayback);    // when ssNoteData and ssTrigData are told to play a step they raise these events for each note/hit to play
  ssTrigData.setHandlerStepPlayback(ssSequencer_handleTrigDataStepPlayback);    // so if the downbeat has a 3 note chord to play and a kick drum hit, ssNoteData handler will get called 3 times and ssTrigData handler once

 // object setups
  ssNoteData.begin();
  ssTrigData.begin();

  ssSettings_begin(true);  // TRUE = load global settings off SD card
  _metro = ssSettings_getGlobalMetroDefault();
  ssClockHelper_attachExtClockInterupt(ssSettings_getGlobalClockSource() > 1); // if clock source setting defaults to an external type turn on CLOCKIN interupt

  ssFileHelper_begin();
  ssClockHelper_begin(ssSettings_getTempo());
  ssMidiHelper_begin();
  ssDisplayUI_begin();
  ssArp_begin();
  ssTuring_begin();

  // ssFileHelper_deleteAllSequenceFiles();
  
  // DEBUG init

}

// **********************************************
// *
// *    MAIN LOOP!!
// *
// **********************************************
void loop() {
  ssClockHelper_advanceClock();       // advances the clock and keeps track of currentstep. Every 1/32 note may noteOn/noteOff notes; a lot of times it returns doing nothing (the space between the ticks/notes)
  ssMidiHelper_processMidi();         // looks to see if any midi INPUT and handles it through events in SSMidiHelper
  ssNoteData.processQueue();         // ssNoteData, the note event store, does some async adds (for things like Fill Bank) and garbage collecting of deletes   
  ssPanel.processPanel();             // scan through the panel buttons looking for presses/releases (knob rotate is done with an interupt and scanning the LEDS is done with a timer interupt; not here)
}

// **********************************************
// *
// *    Pass Thru handler functions
// *
// **********************************************
// attachInterupt will not call Class functions 
// directly so these just pass-thru to the class

void panelTimerInterrupt(){
  ssPanel.handlePanelTimerInterrupt();
}
void panelRotateInterupt(){
  ssPanel.handlePanelKnobInterrupt();
}

// **********************************************
// *
// *    DEBUG functions
// *
// **********************************************
/*
void DEBUG_InitQueue(){
  for (int i=0;i<256;i++){
    DEBUG_Queue[i] = "";
  }
}
void DEBUG_AddToShowQueue(String s){
  DEBUG_Queue[DEBUG_QueueIndex] = s;
  DEBUG_QueueIndex++;
}
void DEBUG_PlayDebugQueue(){
  if (DEBUG_Queue[DEBUG_QueueOutdex] != ""){ 
    Serial.println(DEBUG_Queue[DEBUG_QueueOutdex]);
    DEBUG_Queue[DEBUG_QueueOutdex] = "";
  }
  DEBUG_QueueOutdex++; 
}
void DEBUG_AddPlayDataToQueue(NoteEvent e){
  String s = "";
    s = millis();  
    s +=  "step: ";
    s += e.step;
    s += "  bank: ";
    s += e.bank;
    s += "  track: ";
    s += e.track;
    s += "  cmd: ";
    s += e.cmd;
    s += "  val1: ";
    s += e.val1;
    s += "  val2: ";
    s += e.val2;
    DEBUG_AddToShowQueue(s);
}
*/
// **********************************************
// *
// *    MEMORY DEBUG functions
// *
// **********************************************
extern char _end;
extern "C" char *sbrk(int i);

void freeRam()
{
  char *ramstart = (char *) 0x20070000;
  char *ramend = (char *) 0x20088000;
  char *heapend = sbrk(0);
  register char * stack_ptr asm( "sp" );
  struct mallinfo mi = mallinfo();
  Serial.println(F("Ram used (bytes): "));
  Serial.print(F("  dynamic: ")); Serial.println( mi.uordblks );
  Serial.print(F("  static:  ")); Serial.println(&_end - ramstart );
  Serial.print(F("  stack:   ")); Serial.println(ramend - stack_ptr );
  Serial.print(F("Estimation free Ram: ")); Serial.println(stack_ptr - heapend + mi.fordblks );
}
