#include "Arduino.h"
#include "SSNoteData.h"
#include <stdio.h>

/*
struct NoteEvent{
   int step;
   byte bank;
   byte track;
   byte cmd;
   byte val1;
   byte val2;
   bool skip;
}
*/

/**********************
*
* PUBLIC FUNCTIONS
*
*
**********************/
SSNoteData::SSNoteData() { 
	// constructor
}

void SSNoteData::begin(){
	clearEvents();
  mAddEventQueueIndex = 0;
  mGarbageIndex = 0;
}


void SSNoteData::setHandlerStepPlayback(hStepPlayback h){
	raiseStepPlayback = h;
}
EventCount SSNoteData::getEventUsage(){
	EventCount ec = { mGetEventCount(), NOTEDATABUFFERSIZE };
	return ec;
}
byte SSNoteData::getStepSummary(byte bank, byte track, int step){
  byte rtn = 0;
  NOTEDATABUFFERINDEX stepIndexTop = mEventStepHints[step].top;           // gets index for first row with STEP using hints
  NOTEDATABUFFERINDEX stepIndexBottom = mEventStepHints[step].bottom;
  for (NOTEDATABUFFERINDEX i=stepIndexTop;i<stepIndexBottom;i++){
    NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
    if (e.bank == bank && e.track == track && e.cmd > 0){
      rtn |= 0x01 << (e.cmd - 1) ;   
    }
  }
  return rtn;
}
void SSNoteData::emptyQueue(){
  while (mAddEventQueueIndex > mGetEventCount()){
    mProcessAddEventQueue();
  }
  mGarbageIndex = 0;
  while (mGarbageIndex < mGetEventCount() && mGarbageIndex < NOTEDATABUFFERSIZE){
    mCheckGarbage();
    mGarbageIndex++;  
  }
}
void SSNoteData::processQueue(){
  mProcessAddEventQueue();
  if (mAddEventQueueIndex <= mGetEventCount() ) { mProcessGarbageCollection(); }     // prioritize adding events as they impact sequence; deleteEvents garbage has no impact on sequencer (just takes memory)
}
void SSNoteData::addEvent(NoteEvent e){
	mAddEventData(e, false);
}
void SSNoteData::addEvent(int step, byte bank, byte track, byte cmd, byte val1, byte val2, bool skip){
	NoteEvent e = { step, bank, track, cmd, val1, val2, skip };
	mAddEventData(e, false);
}
void SSNoteData::addEvent(int step, byte bank, byte track, byte cmd, byte val1, byte val2, bool skip, bool top){
	NoteEvent e = { step, bank, track, cmd, val1, val2, skip };
	mAddEventData(e, top);
}

void SSNoteData::playbackStep(byte bank, int step){
	NOTEDATABUFFERINDEX stepIndexTop = mEventStepHints[step].top;						// gets index for first row with STEP using hints
	NOTEDATABUFFERINDEX stepIndexBottom = mEventStepHints[step].bottom;
	// Serial.print(millis());Serial.print(":");Serial.print(step);Serial.print(":");Serial.print(stepIndexTop);Serial.print(",");Serial.println(stepIndexBottom);
	for (NOTEDATABUFFERINDEX i=stepIndexTop;i<stepIndexBottom;i++){
		NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
		if (e.cmd > 0 && e.bank == bank) {
			if (!e.skip) {
				if (raiseStepPlayback != NULL) { raiseStepPlayback(e); }
			} else {
				e.skip = false;
				mEvents[i] = mEventToEVENTDWORD(e);			
			}
		}
	}		
}

// PUBLIC EDIT DATA
void SSNoteData::flipTrackSteps(byte bank, byte track, int startStep, int numOfSteps){
  if (numOfSteps + startStep > NUMOFRECORDERSTEPS) { return; }                    // exceeding bounds

  int top = startStep + numOfSteps;
  int newStep = startStep;
  int addCount = 0;
  for (int s=top - 1;s>=startStep;s--){
    // find where events start/end for current step
    NOTEDATABUFFERINDEX stepIndexTop = mEventStepHints[s].top;                                              // gets index for first row with STEP using hints
    NOTEDATABUFFERINDEX stepIndexBottom = mEventStepHints[s].bottom;
    // go through that steps
    for (NOTEDATABUFFERINDEX i=stepIndexTop;i<stepIndexBottom;i++){                                         // go through that steps
      NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
      // find steps matching bank/track
      if (e.bank == bank && e.track == track && e.cmd != 2){
        int oldStep = e.step;
        e.step = newStep;
        mFastAddEventData(e);
        addCount++;
        if (e.cmd == CMD_NOTEON){
          int partnerEvent = mFindNoteOffPartner(i);
          if (partnerEvent > -1){
            NoteEvent ee = mEVENTDWORDToEvent(mEvents[partnerEvent]);
            ee.step = newStep + (ee.step - oldStep);
            mFastAddEventData(ee);
            addCount++;
            mFastDeleteEventData(partnerEvent);     
          }
        }
        mFastDeleteEventData(i); 
      }
    }
    newStep++;
  }  
  for (int c=0;c<addCount;c++){                                                          // mNudgeEvents does fastDeletes and fastAdds so mEvents doesn't change under the i loop above.  
    mProcessAddEventQueue();                                                              // once done with the i loop (nudge), then process the Adds. The Deletes will get garbage collected later
  }
}
void SSNoteData::nudgeCmdAtStep(byte bank, byte track, int step, byte cmd, int steps){
  NOTEDATABUFFERINDEX stepIndexTop = mEventStepHints[step].top;                                           // gets index for first row with STEP using hints
  NOTEDATABUFFERINDEX stepIndexBottom = mEventStepHints[step].bottom;
  // go through that steps
  int addCount = 0;
  for (NOTEDATABUFFERINDEX i=stepIndexTop;i<stepIndexBottom;i++){                                         // go through that steps
    NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);   
    if (e.bank == bank && e.track == track && e.cmd == cmd){                                              
      mNudgeEvent(i, steps);
      addCount++;
    }
  }
  for (int c=0;c<addCount;c++){                                                          // mNudgeEvents does fastDeletes and fastAdds so mEvents doesn't change under the i loop above.  
    mProcessAddEventQueue();                                                              // once done with the i loop (nudge), then process the Adds. The Deletes will get garbage collected later
  }  
}
void SSNoteData::nudgeStep(byte bank, byte track, int step, int steps){
  NOTEDATABUFFERINDEX stepIndexTop = mEventStepHints[step].top;                                           // gets index for first row with STEP using hints
  NOTEDATABUFFERINDEX stepIndexBottom = mEventStepHints[step].bottom;
  // go through that steps
  int addCount = 0;
  for (NOTEDATABUFFERINDEX i=stepIndexTop;i<stepIndexBottom;i++){                                         // go through that steps
    NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);   
    if (e.bank == bank && e.track == track){                                              
      if (e.cmd == CMD_NOTEON){                                                                      // if a noteOn nudge the partner noteOff
        int partner = mFindNoteOffPartner(i); 
         mNudgeEvent(partner, steps); 
         addCount++;
      }
      mNudgeEvent(i, steps);
      addCount++;
    }
  }
  for (int c=0;c<addCount;c++){                                                          // mNudgeEvents does fastDeletes and fastAdds so mEvents doesn't change under the i loop above.  
    mProcessAddEventQueue();                                                              // once done with the i loop (nudge), then process the Adds. The Deletes will get garbage collected later
  }
}
void SSNoteData::nudgeTrack(byte bank, byte track, int startStep, int numOfSteps, int nudgeSteps){
  if (numOfSteps + startStep > NUMOFRECORDERSTEPS) { return; }                    // exceeding bounds

  int top = startStep + numOfSteps;
  int newStep = startStep;
  int addCount = 0;
  for (int s=top - 1;s>=startStep;s--){
    NOTEDATABUFFERINDEX stepIndexTop = mEventStepHints[s].top;                                           // gets index for first row with STEP using hints
    NOTEDATABUFFERINDEX stepIndexBottom = mEventStepHints[s].bottom;
    // go through that steps
    for (NOTEDATABUFFERINDEX i=stepIndexTop;i<stepIndexBottom;i++){                                         // go through that steps
      NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);   
      if (e.bank == bank && e.track == track){                                              
        if (e.cmd == CMD_NOTEON){                                                                      // if a noteOn nudge the partner noteOff
          int partner = mFindNoteOffPartner(i); 
           mNudgeEvent(partner, nudgeSteps); 
           addCount++;
        }
        mNudgeEvent(i, nudgeSteps);
        addCount++;
      }
    }
  }
  if (addCount > 8) addCount = 8;
  for (int c=0;c<addCount;c++){                                                          // mNudgeEvents does fastDeletes and fastAdds so mEvents doesn't change under the i loop above.  
    mProcessAddEventQueue();                                                             // once done with the i loop (nudge), then process some of the Adds. Do the rest Async. The Deletes will get garbage collected later
  }
}
void SSNoteData::duplicateStepsByTrack(byte bank, byte track, int startStep, int numOfSteps, byte numOfCopies){
	if (numOfSteps * numOfCopies + startStep > NUMOFRECORDERSTEPS) { return; }		                // exceeding bounds

	int top = startStep + numOfSteps;
	clearEventsForTrack(bank,track,top);
		
	for (int s=top - 1;s>=startStep;s--){
		// find where events start/end for current step
		NOTEDATABUFFERINDEX stepIndexTop = mEventStepHints[s].top;						                                  // gets index for first row with STEP using hints
		NOTEDATABUFFERINDEX stepIndexBottom = mEventStepHints[s].bottom;
		// go through that steps
    for (NOTEDATABUFFERINDEX i=stepIndexBottom;i>stepIndexTop;i--){                                         // go through that steps
			NoteEvent e = mEVENTDWORDToEvent(mEvents[i - 1]);
			// find steps matching bank/track
			if (e.bank == bank && e.track == track){
        for (byte c=0;c<numOfCopies;c++){
          e.step += numOfSteps;
				  mFastAddEventData(e);                                                               // e.skip??
        }
			}
		}
	}
}
void SSNoteData::duplicateStepsByBank(byte bank, int startStep, int numOfSteps, byte numOfCopies){
	if (numOfSteps * numOfCopies + startStep > NUMOFRECORDERSTEPS) { return; }		                // exceeding bounds

	int top = startStep + numOfSteps;
	clearEventsForBank(bank,top);
	
  for (int s=top - 1;s>=startStep;s--){
		NOTEDATABUFFERINDEX stepIndexTop = mEventStepHints[s].top;				// find where events start/end for current step
		NOTEDATABUFFERINDEX stepIndexBottom = mEventStepHints[s].bottom;
		for (NOTEDATABUFFERINDEX i=stepIndexBottom;i>stepIndexTop;i--){		// go through that steps
			NoteEvent e = mEVENTDWORDToEvent(mEvents[i - 1]);
			if (e.bank == bank){									                          // find steps matching bank/track
        for (byte c=0;c<numOfCopies;c++){
          e.step += numOfSteps;
          mFastAddEventData(e);                                       // e.skip??
        }                                                             
			}
		}
	}
 Serial.println("finished duplicateStepsByBank");
 Serial.print("addEvent items: ");Serial.println(mAddEventQueueIndex - mGetEventCount());
 
}
void SSNoteData::copyBankTrackToBankTrack(byte srcBank, byte srcTrack, byte dstBank, byte dstTrack){
	clearEventsForTrack(dstBank,dstTrack); 			                                              // delete the destination track in the destination bank
	for (NOTEDATABUFFERINDEX i=mGetEventCount();i>0;i--){
		NoteEvent e = mEVENTDWORDToEvent(mEvents[i - 1]);
		if (e.bank == srcBank && e.track == srcTrack){		                                      // if event match src bank/track
			e.bank = dstBank;								                                                      // change the bank/track to destination
			e.track = dstTrack;
			mFastAddEventData(e);                           		                                  // and add the queue to write the step at destination
		}
	}
}
void SSNoteData::copyBankToBank(byte srcBank, byte dstBank){
  clearEventsForBank(dstBank, 0);                                                           // delete the destination bank			
	for (NOTEDATABUFFERINDEX i=mGetEventCount();i>0;i--){                                                     // add them to the queue backwards, they are processed forwards
		NoteEvent e = mEVENTDWORDToEvent(mEvents[i - 1]);
		if (e.bank == srcBank){
			e.bank = dstBank;
			mFastAddEventData(e);
		}
	}	
}
void SSNoteData::levelCmdValue(byte bank, byte track, int startStep, int numOfSteps, byte cmd, byte cmdVal, byte level, byte mode){
  if (bank > NUMOFBANKS - 1 || track > NUMOFTRACKS - 1 || startStep > NUMOFRECORDERSTEPS-2 || numOfSteps + startStep > NUMOFRECORDERSTEPS || cmd > 7 || level > 127) { return; }
  
  int top = startStep + numOfSteps;

  byte maxValue = 0;
  if (mode == 2) {                                                                            // if normalize, need to run though the steps twice; first time to find the maximum value
    for (int s=startStep;s<top;s++){                                
      NOTEDATABUFFERINDEX stepIndexTop = mEventStepHints[s].top;                                              // find where events start/end for current step
      NOTEDATABUFFERINDEX stepIndexBottom = mEventStepHints[s].bottom;
      for (NOTEDATABUFFERINDEX i=stepIndexTop;i<stepIndexBottom;i++){                                         // go through that steps
        NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
        if (e.bank == bank && e.track == track && e.cmd == cmd){                                // if bank & track match and NoteOn cmd
          int value;
          if (cmdVal == 1){
            value = (int)e.val1;
          } else {
            value = (int)e.val2;
          }
          if (value > maxValue) { maxValue = value; }
        }
      }
    }
  }

  
  for (int s=startStep;s<top;s++){                                
    NOTEDATABUFFERINDEX stepIndexTop = mEventStepHints[s].top;                                              // find where events start/end for current step
    NOTEDATABUFFERINDEX stepIndexBottom = mEventStepHints[s].bottom;
    for (NOTEDATABUFFERINDEX i=stepIndexTop;i<stepIndexBottom;i++){                                         // go through that steps
      NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
      if (e.bank == bank && e.track == track && e.cmd == cmd){                                // if bank & track match and NoteOn cmd
        int value;
        if (cmdVal == 1){
          value = (int)e.val1;
        } else {
          value = (int)e.val2;
        }
        switch (mode){
          case 0:                                                                          // flatten - set everything to the target level
            value = level;                                                                  
          case 1:                                                                          // limit - reduce peaks to limit to target level
            if (value > level) { value = level; }                                          
          case 2:                                                                          // normalize - adjust to make max value == target level
            value += (level - maxValue);                                                   // value adjusted to difference 
        }

        if (value < 0) { value = 0; }
        if (value > 127) { value = 127; }
        
        if (cmdVal == 1){ e.val1 = (byte)value; } else { e.val2 = (byte)value; }
        
        mEvents[i] = mEventToEVENTDWORD(e);
      }
    }
  }
}  

void SSNoteData::adjustCmdValue(byte bank, byte track, int startStep, int numOfSteps, byte cmd, byte cmdVal, int adjustAmount, bool minMaxOut){    // adjustAmmount is positive or negative
  if (bank > NUMOFBANKS - 1 || track > NUMOFTRACKS - 1 || startStep > NUMOFRECORDERSTEPS-2 || numOfSteps + startStep > NUMOFRECORDERSTEPS || cmd > 7 || adjustAmount > 127 || adjustAmount< -127) { return; }

  int top = startStep + numOfSteps;
  for (int s=startStep;s<top;s++){ 
    NOTEDATABUFFERINDEX stepIndexTop = mEventStepHints[s].top;                                              // find where events start/end for current step
    NOTEDATABUFFERINDEX stepIndexBottom = mEventStepHints[s].bottom;
    for (NOTEDATABUFFERINDEX i=stepIndexTop;i<stepIndexBottom;i++){                                         // go through that steps
      NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
      if (e.bank == bank && e.track == track && e.cmd == cmd){                                // if bank & track match and NoteOn cmd
        int newValue;
        if (cmdVal == 1){
          newValue = (int)e.val1 + adjustAmount;
        } else {
          newValue = (int)e.val2 + adjustAmount;
        }
        if (minMaxOut){
          if (newValue < 0)  { newValue = 0; }
          if (newValue > 127){ newValue = 127; }                                              // stay within midi note bounds
        }
        if (newValue > 0 && newValue < 128){                                                  // stay within midi note bounds
          if (cmdVal == 1){ e.val1 = (byte)newValue; } else { e.val2 = (byte)newValue; }
          mEvents[i] = mEventToEVENTDWORD(e);
        }  
      }
    }
  }
}
// PUBLIC DELETE DATA
void SSNoteData::clearEvents(){
	mFullClearEvents();
	mClearHints();
}
void SSNoteData::clearEventsForBank(byte bank){
  clearEventsForBank(bank, 0);
}
void SSNoteData::clearEventsForBank(byte bank, int start){
  for (int s=start;s<mGetLastStep()+1;s++){
    mDeleteBankEventsAtStep(bank, s);
  }
}
void SSNoteData::clearEventsForTrack(byte bank, byte track){
  clearEventsForTrack(bank, track, 0);
}
void SSNoteData::clearEventsForTrack(byte bank, byte track, int start){
  for (int s=start;s<mGetLastStep()+1;s++){                                                 // mGetLastStep is the last used step so we need to delete it too
    mDeleteTrackEventsAtStep(bank, track, s, (start>0));                                    // (start>0) is for delete partners
  }
}
void SSNoteData::clearCmdForTrack(byte bank, byte track, byte cmd){
  clearCmdForTrack(bank, track, cmd, 0);
}
void SSNoteData::clearCmdForTrack(byte bank, byte track, byte cmd, int start){
  for (int s=start;s<mGetLastStep()+1;s++){
    mDeleteCmdEventsAtStep(bank, track, cmd, s);
  }
}
void SSNoteData::clearCmdForBank(byte bank, byte cmd){
  for (int s=0;s<mGetLastStep()+1;s++){
    mDeleteCmdEventsForBankAtStep(bank, cmd, s);
  }    
}
void SSNoteData::eraseAtStep(byte bank, byte track, int step){
	mDeleteTrackEventsAtStep(bank, track, step, true);
}

// FUNCTIONS FOR FILE SAVE/LOAD
void SSNoteData::processPostLoad(){
  mRebuildStepHints();									                                                    // rebuild StepHints after loading mEvents from disk
}
void * SSNoteData::getEventsDataPointer(){				                                          // passed for file save
  return &mEvents;
}
uint32_t  SSNoteData::getEventDataSize(){
  return mGetEventCount() * 4;                                                                // number of events * the sizeof a NOTEEVENTDWORD
}
// END - FUNCTIONS FOR FILE SAVE/LOAD


/**********************
*
* PRIVATE ADD DATA
*
**********************/
void SSNoteData::mAddEventData(NOTEEVENTDWORD edw, bool top){
	mAddEventData(mEVENTDWORDToEvent(edw), top);
}
void SSNoteData::mAddEventData(NoteEvent e, bool top){
  if (e.cmd == 0 || e.step < 0 || e.step > (NUMOFRECORDERSTEPS - 1)) { return; }			    // not a valid event
	
	// convert event to Int32
	NOTEEVENTDWORD eventdw = mEventToEVENTDWORD(e);
	if (top){
		mInsert(eventdw, mEvents, mEventStepHints[e.step].top);
    // mEventStepHints[e.step].top;
	} else {
		mInsert(eventdw, mEvents, mEventStepHints[e.step].bottom);
    // mEventStepHints[e.step].bottom;
	}
	mEventStepHints[e.step].bottom = mEventStepHints[e.step].bottom + 1;
	
	mIncStepHints(e.step);      // increment mEventStepHints for all later events
  mAddEventQueueIndex++;      // increment the queue index so it is corrected
}



void SSNoteData::mFastAddEventData(NoteEvent e){
  if (e.cmd == 0 || e.step < 0 || e.step > (NUMOFRECORDERSTEPS - 1)) { return; }        // not a valid event

  if (mAddEventQueueIndex == 0) { mAddEventQueueIndex = mGetEventCount(); }             // init queue

  NOTEEVENTDWORD eventdw = mEventToEVENTDWORD(e);
  mEvents[mAddEventQueueIndex] = eventdw;

  if (mAddEventQueueIndex < NOTEDATABUFFERSIZE - 1){ mAddEventQueueIndex++; }           // don't overflow the array; TODO handle this better

}
void SSNoteData::mProcessAddEventQueue(){
  if (mAddEventQueueIndex == 0){ return; }                  // don't do anything if the Event list is empty
  
  mAddEventQueueIndex--;                                    // go to next item in queue
  if (mAddEventQueueIndex < mGetEventCount()) {             // if the last item
    mAddEventQueueIndex = mGetEventCount();                 //  set the queue to the bottom of the event table
  } else {
    NOTEEVENTDWORD edw = mEvents[mAddEventQueueIndex];      // grab the event to be added
    if (edw != 0){                                          // if it's not empty
      mEvents[mAddEventQueueIndex] = 0;                     //  clear the queue version of event
      mAddEventData(edw, false);                            //  with "slow" addEvent, mEvents indexes change by +1
    }
  }
}

/**********************
*
* PRIVATE DELETE DATA
*
**********************/

void SSNoteData::mFullClearEvents(){
	for (NOTEDATABUFFERINDEX i=0;i<NOTEDATABUFFERSIZE;i++){
		mEvents[i] = 0;
	}
  mAddEventQueueIndex = 0;
  mGarbageIndex = 0;
}
void SSNoteData::mClearHints(){
	for (int s=0;s<NUMOFRECORDERSTEPS;s++){
		mEventStepHints[s] = {0,0};
	}
}

// DELETE STEPS
void SSNoteData::mDeleteBankEventsAtStep(byte bank, int step){                                    // should I delete partners?
	for (NOTEDATABUFFERINDEX i=mEventStepHints[step].top;i<mEventStepHints[step].bottom;i++){				// go through that steps
		NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
		if (e.bank == bank){									                                                        // find steps matching bank/track
			mFastDeleteEventData(i);
		}
	}
}
void SSNoteData::mDeleteTrackEventsAtStep(byte bank, byte track, int step, bool deletePartner){
  
	for (NOTEDATABUFFERINDEX i=mEventStepHints[step].top;i<mEventStepHints[step].bottom;i++){				// go through events for given step 
		NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
		if (e.bank == bank && e.track == track){									                          // if step event matches bank&track, delete it
																					                                              // if step>0, we are in erase, need to delete NoteOn and partner noteOff
			if (deletePartner && e.cmd == CMD_NOTEON){										                                    // if a noteOn is to be erased we need to find and delete the partner noteOff anywhere in the track
				
				//Serial.print(F("noteOnToBeDeleted"));									                          // another option is to only delete the noteOn and leave the noteOff for
				//Serial.println(i);													                                    // some garbage collecting routine. The noteOffs need to be deleted somewhat quickly
																					                                              // or it may screw up future NoteOn/NoteOff pairs.
        mFastDeleteEventData(i);                                                        // fastDelete the noteOn (set cmd=0; it is still there)
        int partnerEvent = mFindNoteOffPartner(i);
        if (partnerEvent > 0){
				  mFastDeleteEventData(partnerEvent);			
        }														
			} else if (deletePartner && e.cmd == CMD_NOTEOFF){								                          // do nothing. We don't delete noteOffs and their partners because they are either handled above or the 
																					                                              // noteOn is outside the erase window so the partner noteOff doesn't get erased. This means noteOff messages
																					                                              // may survice an erase which is OK.
			} else {
				mFastDeleteEventData(i);												                                // else it is non-note command (ie. CC msg), or we are not in deletePartners (ie we are deleting the entire track), don't bother with partners										
			}
		}
	}	
}
NOTEDATABUFFERINDEX SSNoteData::mFindNoteOffPartner(NOTEDATABUFFERINDEX i){
  int rtn = -1;
  NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
  int si = i;                                                                     // starting from event i
  bool noteOffFound = false;
  while (!noteOffFound){                                                          // loop forward until partner noteOff on is found
    si++;
    if (si >= mGetEventCount()) {                                                 // since we work in loops, wrap around in case noteOff is in front of noteOn
      si = 0; 
    }
    if (si == i) {                                                                // looped in a circle back to i and didn't find anything... oops, bail out
      noteOffFound = true;  
      Serial.println(F("Bailed out of partner search: NoteOff"));                       
    }
    NoteEvent re = mEVENTDWORDToEvent(mEvents[si]);
    if (re.cmd == CMD_NOTEOFF && re.val1 == e.val1) {
      noteOffFound = true;
      rtn =  si;
    }
  }
  return rtn;  
}
void SSNoteData::mDeleteCmdEventsAtStep(byte bank, byte track, byte cmd, int step){	
	for (NOTEDATABUFFERINDEX i=mEventStepHints[step].top;i<mEventStepHints[step].bottom;i++){				// go through that steps
		NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
		if (e.bank == bank && e.track == track && e.cmd == cmd){									// find steps matching bank/track
			mFastDeleteEventData(i);
		}
	}	
}
void SSNoteData::mDeleteCmdEventsForBankAtStep(byte bank, byte cmd, int step){  
  for (NOTEDATABUFFERINDEX i=mEventStepHints[step].top;i<mEventStepHints[step].bottom;i++){       // go through that steps
    NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
    if (e.bank == bank && e.cmd == cmd){                  // find steps matching bank
      mFastDeleteEventData(i);
    }
  } 
}
void SSNoteData::mDeleteEventData(NOTEDATABUFFERINDEX i){
	NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
	
	mRemove(mEvents, i);
	mEventStepHints[e.step].bottom = mEventStepHints[e.step].bottom - 1;
	
	// decrement mEventStepHints for all later events
	mDecStepHints(e.step);	
  if (mAddEventQueueIndex > 0) { mAddEventQueueIndex--; }
}
void SSNoteData::mFastDeleteEventData(NOTEDATABUFFERINDEX i){
  //Serial.println(F("mFastDeleteEventData"));
  //                     sssssssssbbtttccc11111112222222S    sssssssssbbtttccc11111112222222S
  mUpdateEventValue(i, 0b00000000000000000000000000000001, 0b00000000000000111000000000000001);    // set cmd = 0x0, skip = true for given event making it an used event (but valid) that will get garbage collected
}
/**********************
*
* PRIVATE UTILITY FUNCTIONS
*
**********************/
void SSNoteData::mNudgeEvent(NOTEDATABUFFERINDEX i, int steps){
  NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
  int newStep = e.step + steps;
  if (newStep >= NUMOFRECORDERSTEPS) { newStep -= NUMOFRECORDERSTEPS; }
  if (newStep == -1) { newStep == NUMOFRECORDERSTEPS - 1; }
  mFastDeleteEventData(i);                                                            // want to do a fast delete so mEvent order doesn't change
  e.step = newStep;
  mFastAddEventData(e);  
}
bool SSNoteData::mCompareEvents(NOTEEVENTDWORD eventA, NOTEEVENTDWORD eventB, NOTEEVENTDWORD compareMask){
	return (eventA & compareMask) == (eventB & compareMask);
}
void SSNoteData::mUpdateEventValue(NOTEDATABUFFERINDEX i, NOTEEVENTDWORD newValue, NOTEEVENTDWORD updateMask){
	NOTEEVENTDWORD oldValue = mEvents[i];
	mEvents[i] = (~(~oldValue | updateMask)) | (newValue & updateMask);
}
void SSNoteData::mRebuildStepHints(){		// allows me to do bulk edits to Events table without having to update StepHints until the end
	// *** assumes mEvents[] data is continuous (no empty spots)								
	mClearHints();							// reset all hints to zeros
	
	NOTEDATABUFFERINDEX i = 0;
	int lastStepWithEvent = -1;
	NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
	
	while (e.cmd > 0) {
		if (e.step > lastStepWithEvent){
			for (int z=lastStepWithEvent + 1;z<e.step;z++){			// fill in the blanks between steps with cmds
				mEventStepHints[z].top = i;
				mEventStepHints[z].bottom = i;			
			}
		}
		
		if (mEventStepHints[e.step].top == mEventStepHints[e.step].bottom){
			mEventStepHints[e.step].top = i;
			mEventStepHints[e.step].bottom = i + 1;
		} else {
			mEventStepHints[e.step].bottom = i + 1;
		}
		lastStepWithEvent = e.step;

		i++;
		e = mEVENTDWORDToEvent(mEvents[i]);
	}
 
	for (int s=lastStepWithEvent+1;s<NUMOFRECORDERSTEPS;s++){		// fill in any trailing steps
		mEventStepHints[s].top = i;
		mEventStepHints[s].bottom = i;
	}
	
	mGarbageIndex = 0;
}

NOTEDATABUFFERINDEX SSNoteData::mGetEventCount(){      // return first unused index
	int largestBottom = 0;
	largestBottom = mEventStepHints[NUMOFRECORDERSTEPS - 1].bottom;
	return largestBottom;
}
int SSNoteData::mGetLastStep(){       // returns the largest used step value
	NOTEDATABUFFERINDEX lc = mGetEventCount();
	if (lc > 0) { lc--; }
	NoteEvent e = mEVENTDWORDToEvent(mEvents[lc]);
	return e.step;
}
void SSNoteData::mInsert(NOTEEVENTDWORD value, NOTEEVENTDWORD *a, NOTEDATABUFFERINDEX position){
	memmove(a+position+1,a+position,(NOTEDATABUFFERSIZE-position)*4);
	*(a+position) = value;
}
void SSNoteData::mRemove(NOTEEVENTDWORD *a, NOTEDATABUFFERINDEX position){
    memmove(a+position,a+position+1,(NOTEDATABUFFERSIZE-position)*4);
	*(a+(NOTEDATABUFFERSIZE * 4)) = 0;
}
void SSNoteData::mIncStepHints(int startStep){
	for (int s=startStep + 1;s<NUMOFRECORDERSTEPS;s++){
		if (mEventStepHints[s].top < NOTEDATABUFFERSIZE) { mEventStepHints[s].top = mEventStepHints[s].top + 1; }
		if (mEventStepHints[s].bottom < NOTEDATABUFFERSIZE) { mEventStepHints[s].bottom = mEventStepHints[s].bottom + 1; }
	}
}
void SSNoteData::mDecStepHints(int startStep){
	for (int s=startStep + 1;s<NUMOFRECORDERSTEPS;s++){
		if (mEventStepHints[s].top > 0)    { mEventStepHints[s].top = mEventStepHints[s].top - 1; }
		if (mEventStepHints[s].bottom > 0) { mEventStepHints[s].bottom = mEventStepHints[s].bottom - 1; }
	}
}
NOTEEVENTDWORD SSNoteData::mEventToEVENTDWORD(NoteEvent e){
  uint32_t n = 0;
  n =     ((e.step  & 0b111111111) << 23);        // the &'d binary values make sure the value is not beyond the alloed value
  n = n | ((e.bank  & 0b000000011) << 21);
  n = n | ((e.track & 0b000000111) << 18);
  n = n | ((e.cmd   & 0b000000111) << 15);
  n = n | ((e.val1  & 0b001111111) << 8);
  n = n | ((e.val2  & 0b001111111) << 1);
  if (e.skip) { n = n + 0b00000001; }

  return n;
}
NoteEvent SSNoteData::mEVENTDWORDToEvent(NOTEEVENTDWORD d){ //  sssssssssbbtttccc11111112222222S
  NoteEvent e;  
  e.skip  =  d        & 0b00000001;
  e.val2  = (d >> 1)  & 0b01111111;               //  000000000000000000000000002222222
  e.val1  = (d >> 8)  & 0b01111111;               //  000000000000000000000000001111111
  e.cmd   = (d >> 15) & 0b00000111;               //  00000000000000000000000000000cccc
  e.track = (d >> 18) & 0b00000111;               //  000000000000000000000000000000ttt
  e.bank  = (d >> 21) & 0b00000011;               //  0000000000000000000000000000000bb
  e.step  = (d >> 23);                            //  000000000000000000000000sssssssss
  return e;
}

/**********************
*
* PRIVATE ADD/DELETE EVENT QUEUE
*
**********************/

void SSNoteData::mProcessGarbageCollection(){
  mCheckGarbage();
  mGarbageIndex++;
  if (mGarbageIndex >= mGetEventCount() || mGarbageIndex == NOTEDATABUFFERSIZE){
    mGarbageIndex = 0;
  }
}
void SSNoteData::mCheckGarbage(){
  //                                                                               sssssssssbbtttccc11111112222222S
  if (mCompareEvents(mEvents[mGarbageIndex], 0b00000000000000000000000000000001, 0b00000000000000111000000000000001)){        // garbage value is e.cmd = 0x0 and e.skip = 1
    mDeleteEventData(mGarbageIndex);
  }  
}


/**********************
*
* DEBUG
*
**********************/
void SSNoteData::DEBUG_printEventData(int toStep){
  int count = mEventStepHints[toStep].bottom;
	Serial.println(F("------------------START-------------------------------------------"));
	for(int i=0;i<count;i++){
		DEBUG_printSingleEventData(i);
	}
	Serial.println(F("------------------END-------------------------------------------"));
}
void SSNoteData::DEBUG_printSingleEventData(int i){
	NoteEvent e = mEVENTDWORDToEvent(mEvents[i]);
	Serial.print(F("i: "));
	Serial.print( i);
	Serial.print(F("\tstep: "));
	Serial.print( e.step);
	Serial.print(F("  \tbank: "));
	Serial.print( e.bank);
	Serial.print(F(" \ttrack: "));
	Serial.print( e.track);
	Serial.print(F(" \tcmd: "));
	Serial.print( e.cmd);
	Serial.print(F(" \tval1: "));
	Serial.print( e.val1);
	Serial.print(F(" \tval2: "));		
	Serial.print( e.val2);
	Serial.print(F("   \tskip: "));
	if (e.skip) {
    if (e.cmd == 0) {
      Serial.print(F("G"));
    } else {
		  Serial.print(F("S"));
    }
	} else {
		Serial.print(F("-"));
	}

	Serial.print(F("\t\t| hintStep: "));		
	Serial.print( i);
	Serial.print(F("\tvalue: "));		
	Serial.print( mEventStepHints[i].top);
	Serial.print(F(","));
	Serial.println( mEventStepHints[i].bottom);	
}
void SSNoteData::DEBUG_printNumOfEvents(){
	Serial.print(F("Events used: "));
	Serial.print(mGetEventCount());
	Serial.print(F("/"));
	Serial.println(NOTEDATABUFFERSIZE);
}
int SSNoteData::DEBUG_getEventQueueCount(){
  if (mAddEventQueueIndex == 0){ return 0; } else { return mAddEventQueueIndex - mGetEventCount(); }
}
