// **********************************************
// *
// *    MIDI helpers 
// *
// **********************************************
using namespace midi;


uint16_t mKeyInputNoteState[128];                     // 128 midi noteOns input from input keyboard/controller
byte mKeyOutputNoteState[128][32];                    // 128 midi notes, 32 channels keyboard notes;  COUNT of midi out notes played by keyboard
byte mSeqOutputNoteState[128][32];                    // 128 midi notes, 32 channels sequencer notes; COUNT midi out notes played by sequencer
byte mSeqOutputTrackState[8];                         // a count of SEQ notes active on a track
byte mLastPitchBend = 64;
byte mLastPatchValue[8] = {255,255,255,255,255,255,255,255};

//*
//*
//* SEND MIDI DATA TO MIDI LIBRARY
//*
//*
void ssMidiHelper_sendTrackNote(byte track,  byte note, byte vel, bool noteOn, byte type){
  if ((noteOn == ON) && (type == SEQ) && (_panelPage == DEFAULT) && !_showSuperTriggerUI && (ssSettings_getRedLedDisplay() & 0b10)){ ssPanel.fadePoint(ROW_RED,track); }   // flash the red LEDs to show tracks playing notes
 
  for (byte c=0;c<32;c++){                                  // check the MIDI channel mapping to see what midi channels this track plays on 
    uint32_t mask = 0x80000000 >> c; 
    if (ssSettings_getMidiTrackMap(track) & mask){
      sendMidiNote(c + 1, note, vel, noteOn, type); 
    }
  }
  if (type == SEQ){                                         // "log" that a note is being played or not-played from a track
    if (noteOn) {
      mSeqOutputTrackState[track]++;
    } else {
      mSeqOutputTrackState[track]--;
    }
  }
  
  if (_inKeyboard && !_inTrigger && track == _currentNoteTrack && type == SEQ){
    ssDisplay_displayMidiNote(note, noteOn); 
  }
}
void ssMidiHelper_sendTriggerTrackHit(byte track, bool accent){                         // maps trigger hits through drum map to get midi note
  byte trackLed = 8 + track;
  if (_showSuperTriggerUI) { trackLed = track;};
  
  if ((_panelPage == DEFAULT) && (trackLed<16) && (ssSettings_getRedLedDisplay() & 0b10)){ ssPanel.fadePoint(ROW_RED,trackLed); }   // flash the red LEDs to show tracks playing notes
 
  byte vel;
  if (accent) {
    vel = ssSettings_getMidiTriggerNoteMap(track).accent;
  } else {
    vel = ssSettings_getMidiTriggerNoteMap(track).velocity;
  }
  sendTriggerNote(track,ssSettings_getMidiTriggerNoteMap(track).note, vel);
  if (track > 12 && track < 16){                  // this is the trigger out
    ssPanel.togglePort(track - 12);               // 1 - 3 
  }
}
void ssMidiHelper_sendTriggerTrackDirectNote(byte track, byte note, bool accent){       // uses specified note for midi note
  byte vel;
  if (accent) {
    vel = ssSettings_getMidiTriggerNoteMap(track).accent;
  } else {
    vel = ssSettings_getMidiTriggerNoteMap(track).velocity;
  }
  sendTriggerNote(track, note, vel);
}
void ssMidiHelper_sendTrackPitchBend(byte track, int value){
  for (byte c=0;c<32;c++){
    uint32_t mask = 0x80000000 >> c; 
    if (ssSettings_getMidiTrackMap(track) & mask){
      if (c < 16) {
        MIDI.sendPitchBend(value - 8192,c+1);  
      } else {
        MIDI2.sendPitchBend(value - 8192,c-15);       
      }
    }
  }
}
void ssMidiHelper_sendTrackPatchChange(byte track, byte data){
  for (byte c=0;c<32;c++){
    uint32_t mask = 0x80000000 >> c; 
    if (ssSettings_getMidiTrackMap(track) & mask){
      if (c < 16) {
        MIDI.sendProgramChange(data,c+1);  
      } else {
        MIDI2.sendProgramChange(data,c-15);       
      }
    }
  }
  mLastPatchValue[track] = data;   
}
void ssMidiHelper_sendTrackCCMsg(byte track, byte msg, byte value){
  for (byte c=0;c<32;c++){
    uint32_t mask = 0x80000000 >> c; 
    if (ssSettings_getMidiTrackMap(track) & mask){
      if (c<16){
        MIDI.sendControlChange(msg, value, c+1);  
      } else {
        MIDI2.sendControlChange(msg, value, c-15);   
      }
    }
  }
}

// PRIVATE play MIDI note
void sendMidiNote(byte channel, byte note, byte vel, bool noteOn, byte type){
  if ((note < 128) && (channel < 33)) { 
    if (noteOn) {
      if (channel < 17){
        MIDI.sendNoteOn(note,vel,channel);
      } else {
        MIDI2.sendNoteOn(note,vel,channel - 16);
      }
    } else {
      if (channel < 17){
        MIDI.sendNoteOff(note,vel,channel);
      } else {
        MIDI2.sendNoteOff(note,vel,channel - 16);
      }
    }
    if (type == LIVE) { 
      mKeyOutputNoteState[note][channel - 1] = noteOn;
    } else {  // SEQ
      if (noteOn){
        mSeqOutputNoteState[note][channel - 1]++;
      } else {
        if (mSeqOutputNoteState[note][channel - 1] > 0) { mSeqOutputNoteState[note][channel-1]--; }
      }
    }
    //ssMidiHelper_DEBUG_showNoteState();
  }
}
// PRIVATE play trigger note
void sendTriggerNote(byte track, byte note, byte velocity){
  for (byte c=0;c<32;c++){        // TODO - perf hit?
    uint32_t mask = 0x80000000 >> c; 
    if (ssSettings_getMidiTriggerMap(track) & mask){
      sendMidiNote(c + 1,note,velocity,true,LIVE);
      sendMidiNote(c + 1,note,0,false,LIVE); 
    }
  }
  
}



//*
//*
//* HANDLE MIDI LIBRARY EVENTS
//*
//*
// handles MIDI in events from midi library; massages data and passes to Sequencer and/or Robots
void ssMidiHelper_handleMidiNoteOnPortA(byte channel, byte note, byte velocity){
  // figure out which handler to use
  KeyInputState kis;

  kis.note = note;
  kis.velocity = velocity;
  kis.bank = _currentBank;
  kis.recorded = false;         // this is determined later by the handler
  kis.port = 0;  
  ssMidiHelper_handleMidiNoteOn(kis);
}
void ssMidiHelper_handleMidiNoteOnPortB(byte channel, byte note, byte velocity){
  // figure out which handler to use
  KeyInputState kis;

  kis.note = note;
  kis.velocity = velocity;
  kis.bank = _currentBank;
  kis.recorded = false;         // this is determined later by the handler
  kis.port = 1;  
  ssMidiHelper_handleMidiNoteOn(kis);
}
void ssMidiHelper_handleMidiNoteOn(KeyInputState kis){

  if (_inTrigger) {
    kis.handler = TRIGGER; 
    kis.track = _currentTriggerTrack;
  }
  else if (_inStepRec){
    kis.handler = STEPREC;
    kis.track = _currentNoteTrack;
  }
  else if (ssArp_onArpTrack()) {
    kis.handler = ARP; 
    kis.track = _currentNoteTrack;
  }
  else if (ssTuring_onTuringTrack()){
    kis.handler = TURING; 
    kis.track = _currentNoteTrack;
  }
  else if (ssChord_onChordTrack()){
    kis.handler = CHORD;
    kis.track = _currentNoteTrack;
  }
  else {
    kis.handler = SEQUENCER; 
    kis.track = _currentNoteTrack;
  }

  setKeyInputNoteState(kis);                    // store the noteOn (keydown) and how/where it was used
  
  // send it to the handler
  switch (kis.handler){
    case SEQUENCER:
      ssSequencer_handleMidiNoteOn(kis);  
      break;
    case TRIGGER:
      ssSequencer_handleMidiNoteOn(kis);        // sssequencer handles TRIGGER midi notes for now; may change
      break;
    case ARP:
      ssArp_handleMidiNoteOn(kis);
      break;
    case TURING:
      ssTuring_handleMidiNoteOn(kis);
      break;
    case CHORD:
      ssChord_handleMidiNoteOn(kis);
      break;
    case STEPREC:
      ssSequencer_handleStepRecMidiNoteOn(kis);
      break;
  }
}
void ssMidiHelper_handleMidiNoteOff(byte channel, byte note, byte velocity){

  KeyInputState kis = ssMidiHelper_getKeyInputNoteState(note);
  
  // send it to the handler
  switch (kis.handler){
    case SEQUENCER:
      ssSequencer_handleMidiNoteOff(kis);
      break;  
    case TRIGGER:
      ssSequencer_handleMidiNoteOff(kis);        // sequencer handles TRIGGER midi notes for now; may change
      break;
    case ARP:
      ssArp_handleMidiNoteOff(kis);
      break;
    case TURING:
      ssTuring_handleMidiNoteOff(kis);
      break;
    case CHORD:
      ssChord_handleMidiNoteOff(kis);
      break;
    case STEPREC:
      ssSequencer_handleStepRecMidiNoteOff(kis);
      break;
    //case INVALID:                              // should not happen, means the noteOn was not recorded
      // do nothing
      //break;
  }  

  clearKeyInputNoteState(note);                 // once the handlers have dealt with the noteOff clear the inputState
}
void ssMidiHelper_handlePitchBend(byte channel, int value){
  int lowResValue = (value + 8192)/128;
  switch (ssSettings_getRecPitchbendResolution()){
    case 0:     // no pitch bend -- still pass it through so it is played live
      ssSequencer_handlePitchBend(channel,value + 8192);
      break;
    case 1:     // low resolution
      if (lowResValue != mLastPitchBend || value == 0){
        ssSequencer_handlePitchBend(channel,lowResValue * 128);
        mLastPitchBend = lowResValue;
      }
      break;
    case 2:     // high resolution   
      ssSequencer_handlePitchBend(channel,value + 8192);
      break;   
  }
}
void ssMidiHelper_handleCC(byte channel, byte msg, byte value){
  ssSequencer_handleCC(channel, msg, value); 
}



//*
//*
//* TWEAK CURRENT MIDI STATE
//*
//*
void ssMidiHelper_killAllOutNotes(byte type){    // 1 = live, 2 = sequencer, 3 = both
  for (byte n=0;n<128;n++){
    for (byte c=0;c<32;c++){
      if (type & 1){
        if (mKeyOutputNoteState[n][c]) { sendMidiNote(c + 1,n,0,false,LIVE);} 
      } 
      if (type & 2){
        byte t = mSeqOutputNoteState[n][c];
        for (byte i =0;i<t;i++){
          sendMidiNote(c + 1,n,0,false,SEQ);
        } 
      }
    }
  if (n<8 && type == SEQ) { mSeqOutputTrackState[n] = 0; }
  }
}
void ssMidiHelper_killActiveSeqTracks(){
  for (byte t=0;t<8;t++){
    if (mSeqOutputTrackState[t] > 0){
      ssMidiHelper_killTrackOutNotes(t,SEQ);
    }
  }
}
void ssMidiHelper_killTrackOutNotes(byte track, byte type){
  for (byte c=0;c<32;c++){
    uint32_t mask = 0x80000000 >> c;
    // if (negateTrack) { mask = ~mask; }
    if (ssSettings_getMidiTrackMap(track) & mask){
      for (byte b=0;b<128;b++){
        if (type & LIVE){                                                               // if type is BOTH then both LIVE and SEQ will be true
          if (mKeyOutputNoteState[b][c]) { sendMidiNote(c + 1,b,0,false,LIVE);} 
        } 
        if (type & SEQ){
          byte t = mSeqOutputNoteState[b][c];
          for (byte i =0;i<t;i++){
            sendMidiNote(c + 1,b,0,false,SEQ);
          } 
        }
      } 
    }
  }
  mSeqOutputTrackState[track] = 0;
}
void ssMidiHelper_transposePlayingSeqNotesForTrack(byte track, int halfSteps){
  for (byte c=0;c<32;c++){
    uint32_t mask = 0x80000000 >> c; 
    if (ssSettings_getMidiTrackMap(track) & mask){
      for (byte b=0;b<128;b++){
        int nv = b + halfSteps;
        if (nv > -1 && nv < 128){
          byte t = mSeqOutputNoteState[b][c];
          for (byte i =0;i<t;i++){
            sendMidiNote(c + 1,b,0,false,SEQ);
            sendMidiNote(c + 1,(byte)nv,0,true,SEQ);
          } 
        }
      } 
    }
  }
}

/*
 * 
 * PATCH SELECT PAGE
 * 
 * 
 */

void ssMidiHelper_showPatchSelectPage(){
  _currentPatchBank = 0;
  ssPanel.display.setCursor(0,0);
  ssPanel.display.print(str_SELECTPATCH);
  ssMidiHelper_updatePatchSelectPage();
  for (byte b=0;b<4;b++){
    ssPanel.setPoint(ROW_BANKAUX,SOLID,b,OFF);      
    ssPanel.setPoint(ROW_BANKAUX,FASTBLINK,b,OFF);
  }
} 
void ssMidiHelper_updatePatchSelectPage(){
  ssPanel.display.setCursor(0,1);
  ssPanel.display.print("<");

  if (mLastPatchValue[_currentNoteTrack] < 128){
    char tmp[4];
    itoa(mLastPatchValue[_currentNoteTrack] + 1,tmp,10);
    ssPanel.display.print(tmp);
  } else {
    ssPanel.display.print(str_UNKNOWN);
  }
  ssPanel.display.print("   ");
  ssPanel.display.setCursor(4,1);
  ssPanel.display.print("> [ ");
  ssPanel.display.setCursor(8,1);

  char tmp2[4];
  itoa(_currentPatchBank * 16 + 1,tmp2,10);
  ssPanel.display.print(tmp2);
  ssPanel.display.print("-");
  itoa(_currentPatchBank * 16 + 16,tmp2,10);
  ssPanel.display.print(tmp2);
  ssPanel.display.print("   ");
  
  ssPanel.display.setCursor(15,1);
  ssPanel.display.print("]"); 
  showPatchSelectBankLEDs();
}
void ssMidiHelper_handlePatchChangeKnobRotate(int delta){
  mLastPatchValue[_currentNoteTrack] += delta;
  if (mLastPatchValue[_currentNoteTrack] > 127) { mLastPatchValue[_currentNoteTrack] -= 128; }
  ssMidiHelper_handlePatchChange(mLastPatchValue[_currentNoteTrack], false);
  ssMidiHelper_updatePatchSelectPage();
}
void showPatchSelectBankLEDs(){
  for (byte b=0;b<4;b++){
    ssPanel.setPoint(ROW_BANKAUX,SOLID,b,OFF); 
    if (_currentPatchBank%4 == b){
      if (_currentPatchBank < 4){
        ssPanel.setPoint(ROW_BANKAUX,BLINK,b,ON); 
      } else {
        ssPanel.setPoint(ROW_BANKAUX,FASTBLINK,b,ON); 
      }
    } else {
      ssPanel.setPoint(ROW_BANKAUX,BLINK,b,OFF);
      ssPanel.setPoint(ROW_BANKAUX,FASTBLINK,b,OFF);      
    }
  }
}
void ssMidiHelper_handlePatchChange(byte patch, bool showPopup){
  if (!_inTrigger){
    ssMidiHelper_sendTrackPatchChange(_currentNoteTrack,patch);
    char outStr[17];
    strcpy(outStr, str_PATCHSENT);
    char patchStr[4];
    itoa(patch+1,patchStr,10);
    strcat(outStr, patchStr);   

    if (showPopup) { ssDisplayUI_showPopupMessage(str_SELECTPATCH, outStr,3, PATCHSELECT); }
    // patch changes are recorded by Mark Patch
  }  
}
void ssMidiHelper_markPatch(){
  if (_inTrigger){ return; }                                                  // can only mark patch when on a note track
  if (mLastPatchValue[_currentNoteTrack] > 127) {                             // can only mark the patch if is has been set
    ssDisplayUI_showPopupMessage(str_ERRORNO,str_PATCHSELECTED,3);
    return;   
  }
  
  int stp =  _currentStep;                                                    // figure out where to put the mark
  if (stp == -1) { stp = 0; }
  bool skp = false;
  if (ssClockHelper_getOffset() > 4){
    stp++;
    if (stp == ssSettings_getBankLength(_currentBank) * 16) { stp = 0; }
    skp = true;
  }
  ssNoteData.addEvent(stp*2,_currentBank,_currentNoteTrack,4,mLastPatchValue[_currentNoteTrack],0, skp, true);     // add the patch mark to the top of the step
  char patchStr[4];
  itoa (mLastPatchValue[_currentNoteTrack]+1,patchStr,10);
  char stpStr[4];
  itoa(stp,stpStr,10);
  char outStr1[17];
  strcpy(outStr1, str_PATCHMARKED1);
  strcat(outStr1, patchStr);
  char outStr2[17];
  strcpy(outStr2, str_PATCHMARKED2);
  strcat(outStr2, stpStr);
  ssDisplayUI_showPopupMessage(outStr1,outStr2,3); 
  ssFileHelper_makeFileDirty(8);
}
//*
//*
//* OTHER STUFF
//*
//*
byte ssMidiHelper_getKeyInputNoteStateCount(){
  byte count = 0;
  for (byte c=0;c<128;c++){
    if (mKeyInputNoteState[c]>0) count++; 
  }
  return count;
}
void clearKeyInputNoteState(byte note){
  mKeyInputNoteState[note] = 0;  
}
void setKeyInputNoteState(KeyInputState kis){      //  0000000RHHHBBTTT  handler, bank and track noteOn was assigned to
  mKeyInputNoteState[kis.note] = ((kis.handler & 0b111) << 5) + ((kis.bank & 0b11) << 3) + (kis.track & 0b111);
  if (kis.recorded){
    ssMidiHelper_setKeyInputNoteStateRecBit(kis.note);  
  } else {
    clearKeyInputNoteStateRecBit(kis.note);
  }
} 
KeyInputState ssMidiHelper_getKeyInputNoteState(byte note){
  KeyInputState rtn;
  rtn.note = note;
  rtn.velocity = 0xFF;     // undefined because it is not stored
  rtn.handler = (mKeyInputNoteState[note] & 0b011100000) >> 5;
  rtn.bank = (mKeyInputNoteState[note] & 0b000011000) >> 3;
  rtn.track = (mKeyInputNoteState[note] & 0b000000111);
  rtn.recorded = getKeyInputNoteStateRecBit(note);
  return rtn;
}
void ssMidiHelper_setKeyInputNoteStateRecBit(byte note){
  uint16_t tmpKis = mKeyInputNoteState[note];
  mKeyInputNoteState[note] = tmpKis | 0b100000000;
}
void clearKeyInputNoteStateRecBit(byte note){
  mKeyInputNoteState[note] = ~(~mKeyInputNoteState[note] | 0b100000000);
}
bool getKeyInputNoteStateRecBit(byte note){
  return (mKeyInputNoteState[note] & 0b100000000) >> 8;
}

// out going MIDI
void sendMidiClock(){
  MidiType t = Clock;
  MIDI.sendRealTime(t);
  MIDI2.sendRealTime(t);
}

void sendMidiStart(){
  MidiType t = Start;
  MIDI.sendRealTime(t);  
  MIDI2.sendRealTime(t);  
}

void sendMidiStop(){
  MidiType t = Stop;
  MIDI.sendRealTime(t); 
  MIDI2.sendRealTime(t); 
}

void sendMidiCont(){
  MidiType t = Continue;
  MIDI.sendRealTime(t);  
  MIDI2.sendRealTime(t);  
}
void ssMidiHelper_handleClock(){
  if (ssSettings_getGlobalClockSource() != 1){ return; }
  ssClockHelper_handleMidiClockIn();
  
}
void ssMidiHelper_handleStart(){
  if (ssSettings_getGlobalClockSource() != 1){ return; }
  ssClockHelper_handleMidiStart();
}
void ssMidiHelper_handleContinue(){
  if (ssSettings_getGlobalClockSource() != 1){ return; }
  ssClockHelper_handleMidiContinue();
}
void ssMidiHelper_handleStop(){
  if (ssSettings_getGlobalClockSource() != 1){ return; }
  ssClockHelper_handleMidiStop();
}

void ssMidiHelper_begin(){
  MIDI.setHandleNoteOn(ssMidiHelper_handleMidiNoteOnPortA);
  MIDI.setHandleNoteOff(ssMidiHelper_handleMidiNoteOff);
  MIDI.setHandlePitchBend(ssMidiHelper_handlePitchBend);
  MIDI.setHandleControlChange(ssMidiHelper_handleCC);
  MIDI2.setHandleNoteOn(ssMidiHelper_handleMidiNoteOnPortB);
  MIDI2.setHandleNoteOff(ssMidiHelper_handleMidiNoteOff);
  MIDI2.setHandlePitchBend(ssMidiHelper_handlePitchBend);
  MIDI2.setHandleControlChange(ssMidiHelper_handleCC);
  MIDI.setHandleClock(ssMidiHelper_handleClock);
  MIDI.setHandleStart(ssMidiHelper_handleStart);
  MIDI.setHandleContinue(ssMidiHelper_handleContinue);
  MIDI.setHandleStop(ssMidiHelper_handleStop);
  
  MIDI.begin(ssSettings_getMidiInChannel()); // TODO - MIDI in modes (1 - 16, omni, MAPPED); currently 1 - 16 from setting
  MIDI.setThruFilterMode(midi::Thru::DifferentChannel);

  MIDI2.begin(MIDI_CHANNEL_OMNI); // TODO - MIDI in modes (1 - 16, omni, MAPPED); currently 1 - 16 from setting
  MIDI2.setThruFilterMode(midi::Thru::Off);

  // init state arrays
  for (byte b=0;b<128;b++){
    mKeyInputNoteState[b] = 0;      // zero means handler is 0 which is INVALID
    for (byte c=0;c<32;c++){
      mKeyOutputNoteState[b][c] = 0;
      mSeqOutputNoteState[b][c] = 0;
    }
  }
}
void ssMidiHelper_processMidi(){
  MIDI.read();  
  MIDI2.read();  
}
void ssMidiHelper_updateMidiInputChannel(){
  ssMidiHelper_killAllOutNotes(LIVE);
  MIDI.setInputChannel(ssSettings_getMidiInChannel());
}
byte ssMidiHelper_getKeyNoteState(byte n, byte c){
  return mKeyOutputNoteState[n][c];  
}
byte ssMidiHelper_getSeqNoteState(byte n, byte c){
  return mSeqOutputNoteState[n][c];  
}

// *******************************
//
//       DEBUG
//
// *******************************

void ssMidiHelper_DEBUG_showNoteState(){
    String s1 = "LIVE:";
    String s2 = " SEQ:";
    for (byte n=0;n<128;n++){
      if (mKeyOutputNoteState[n][_currentNoteTrack] > 0 ) { s1 += mKeyOutputNoteState[n][0]; } else { s1 += "_"; }
      if (mSeqOutputNoteState[n][_currentNoteTrack] > 0 ) { s2 += mSeqOutputNoteState[n][0]; } else { s2 += "_"; }
    }
    Serial.print(s1);
    Serial.println(s2);
}
void ssMidiHelper_DEBUG_showDrumMap(){
  Serial.println(F("DRUM MAP--------------------------------"));
  TriggerNoteMap tdm;
  for (byte i=0;i<16;i++){
    tdm = ssSettings_getMidiTriggerNoteMap(i);
    Serial.print(tdm.note);
    Serial.print(F(","));
    Serial.print(tdm.velocity);
    Serial.print(F(","));
    Serial.print(tdm.accent);
    Serial.println();
  }
  Serial.println(F("END DRUM MAP----------------------------"));
}

void ssMidiHelper_panic(){
  //Serial.print("panic...");
  ssClockHelper_stop();
  for (byte c=1;c<17;c++){
    for (byte n=0;n<128;n++){
      MIDI.sendNoteOff(n,0,c);  
      MIDI2.sendNoteOff(n,0,c);  
    }
  }
  //Serial.println("over");
  ssClockHelper_start();
}

void ssMidiHelper_ssMidiHelper_DEBUG_showKis(KeyInputState kis){
  Serial.print(F("KIS: "));
  Serial.println(mKeyInputNoteState[kis.note], BIN);
}
