/*
 * SSClockHelper 
 *    wraps SSClock class which is the internal clock - no other code should call SSClock directly
 *    handles SSClock events
 *    handles MIDI CLOCK IN events
 *    handles EXT TRIGGER CLOCK IN
 *    manages _currentStep which the current 1/16 step as displayed by the red LEDs
 *    manages transport state - play/record/stop
 *    raises transport events
 *    magaes transport button LEDs
 *    manages tempo including handling knob turns for tempo and displaying tempo
 *    manages swing including UI pages with knob turn, etc
 *    handles count in functionality
 *    
 * SSClock is the internal clock class
 * 
 */

SSClock ssClock;

// **********************************************
// *
// *    CURRENT STEP MANAGEMENT
// *
// **********************************************

void ssClockHelper_advanceNextSequencerStep(){            // advances sequencer to next step
  ssClockHelper_advanceSequencerStep(1);  
}
void ssClockHelper_advanceSequencerStep(int steps){       // advances sequencer N number of steps
  
  // if (_currentStep + steps >= ssSettings_getBankLength(_currentBank) * 16) { ssSequencer_handleBeforeLoop(_inRec); }    // test if we are going to loop
  
  _currentStep += steps;
  ssClockHelper_handleWrapAround();
  ssDisplayUI_updateStepDisplay();    
}
void ssClockHelper_resetSequencerStep(){                 // resets the sequencer steps to stopped state of -1
  _currentStep = -1;
}
void ssClockHelper_restartSequencerStep(){               // starts playback back at the beginning; first played step is at 0
  _currentStep = 0;
}

byte mSegment = 0;
int mSegDirection = 1;


void ssClockHelper_handleWrapAround(){
  if (_activeRobot != 3) {
    if (_currentStep >= ssSettings_getBankLength(_currentBank) * 16) {
      _currentStep -= ssSettings_getBankLength(_currentBank) * 16; 
    } 
  } else {
    if (_currentStep % (ssSettings_getBankLength(_currentBank) * 16) == 0) {
      byte numberofSegements = 16 / ssSettings_getBankLength(_currentBank);
      if (numberofSegements < 2) { gotoSegment(0); return; }
      gotoSegment(mSegment);

      //Serial.print("1mSegment:");Serial.println(mSegment);
      
      // prep mSegment for the next loop; this way we always start at segment 0
      
      RobotSettings rsSettings = ssSettings_getRobotSettings();                     // Direction         - UP(0), DOWN(1), INCLUDE(2), EXCLUDE(3), RANDOM(4), RANDOMWEIGHTED (5)
      //Serial.print("rsSettings.direction");Serial.println(rsSettings.direction);

      switch (rsSettings.direction){
        case 0: // up
          mSegment++;
          if (mSegment >= numberofSegements) { mSegment = 0; }
          break;
        case 1: // down
          if (mSegment == 0) { mSegment = numberofSegements; }
          mSegment--;
          break;
        case 2: // include 
          if (mSegDirection == 1){
            mSegment++;
            if (mSegment >= numberofSegements) { mSegment--; mSegDirection = -1; }         
          } else {
            if (mSegment == 0) { mSegment = 1; mSegDirection = 1;}
            mSegment--;            
          }
          break;
        case 3: // exclude 
          if (mSegDirection == 1){
            mSegment++;
            if (mSegment >= numberofSegements) { mSegment-= 2; mSegDirection = -1; }         
          } else {
            if (mSegment == 0) { mSegment = 2; mSegDirection = 1;}
            mSegment--;            
          }
          break;
        case 4:
          mSegment = random(numberofSegements);
          break;
        case 5:
          if (random(4)){
            mSegment = 0;  
          } else {
            mSegment = random(numberofSegements);
          }
          break;
      }
      //Serial.print("2mSegment:");Serial.println(mSegment);
    }
  }
}
void gotoSegment(byte s){
  ssClockHelper_jumpToBar(ssSettings_getBankLength(_currentBank) * s);
}
void ssClockHelper_jumpToBar(byte b){
  ssClockHelper_jumpToStep(b * 16);
}
void ssClockHelper_advanceBar(int b){                                           // can be negative to go back; handles wrap around
  byte tmpBar = _currentStep /16;                                               // set it to the current bar
  //Serial.print("current bar:");Serial.println(tmpBar);
  tmpBar = (byte)(tmpBar + b) % ssSettings_getBankLength(_currentBank);               // advance or go back
  //Serial.print("new bar:");Serial.println(tmpBar);
  ssClockHelper_jumpToBar(tmpBar);
}
void ssClockHelper_jumpToStep(byte s){
  if (s >= MAXSTEPS) { _currentStep = 0; }
  _currentStep = s;
  ssMidiHelper_killActiveSeqTracks();       // TODO: ? should this be a "fastforward" NoteOff only to turn off any skipped over noteOffs
  if (!_inPlay && s==0 && _panelPage != STEPEDIT) { _currentStep = -1; }         // if stopped re-adjust
}
void ssClockHelper_jumpToSubStep(byte s){
  ssMidiHelper_killActiveSeqTracks();       // TODO: ? should this be a "fastforward" NoteOff only to turn off any skipped over noteOffs
  int stp = _currentStep;                   // TODO: this -1 step when stopped seems hacky
  if (stp == -1) { stp = 0; }                // not inPlay and stopped, adjust so the math works
  stp = (stp & 0xF0) + s;                   // keep the same bar (0xF0) but adjust the step (_currentStep runs at 256 while ssNoteData runs at 512 steps)
  if (!_inPlay && stp==0) { stp = -1; }     // if stopped re-adjust to -1 step
  _currentStep = stp;
}

void ssClockHelper_makeActiveRobot(){
  if (_activeRobot != 3){             // if not already active robot
    _activeRobot = 3;
  }  
}
void ssClockHelper_patternModeOff(){
  _activeRobot = 0xFF;
}
// **********************************************
// *
// *    TRANSPORT CONTROL
// *
// **********************************************
void ssClockHelper_togglePlayTransport(){                 // toggles between play and pause state
  _inPlay = !_inPlay;
  ssClockHelper_setPlayTransport(_inPlay);
}
void ssClockHelper_setPlayTransport(bool inPlay){   
  ssClockHelper_setPlayTransport(inPlay, true);    
}
void ssClockHelper_setPlayTransport(bool inPlay, bool useClock){         // actually does the play/pause state change (_inPlay) and notifies other systems of the state change
  _inPlay = inPlay;
  if (inPlay) {
    if (useClock) { ssClockHelper_start(); }
  } else {
    ssPanel.setClick(OFF); // turn off metro
    if (useClock) { ssClockHelper_stop(); }
  }
  
  ssSequencer_handlePlay(inPlay);
  ssArp_handlePlay(inPlay);
  ssChord_handlePlay(inPlay);
  ssDisplay_handlePlay(inPlay);
  

  ssClockHelper_setPlayLED(inPlay);  
}
void ssClockHelper_toggleRecordTransport(){               // toggle Record state
  _inRec = !_inRec;                   
  ssClockHelper_setRecordTransport(_inRec);
}
void ssClockHelper_setRecordTransport(bool inRec){        // actually sets the record state (_inRec)
  _inRec = inRec;
  if (inRec) {
    ssSequencer_exitStepRecord();     // exit step record
    ssClockHelper_clearCountIn();     // cancel count in
  } else {
    ssPanel.setClick(OFF);            // turn off metro
  }

  ssSequencer_handleRec(inRec);      // throw events for other systems
  ssArp_handleRec(inRec);
  ssChord_handleRec(inRec);

  ssClockHelper_setRecLED(inRec);
}
void ssClockHelper_stopTransport(){
  ssClockHelper_stopTransport(true);
}
void ssClockHelper_stopTransport(bool useClock){
  ssClockHelper_setPlayTransport(OFF, useClock);             // stop the clock
  ssClockHelper_setRecordTransport(OFF);
  ssClockHelper_resetSequencerStep();              // reset currentStep to -1
  
  ssSequencer_handleStop();                        // let everyboby know we stopped
  ssArp_handleStop();
  ssChord_handleStop();
  ssTuring_handleStop();
  ssPanel.setClick(OFF);
  ssDisplayUI_handleStop();      
}
void ssClockHelper_setTempo(float t){
  ssClock.setTempo(t);                             // change the clock tempo
  ssSettings_setTempo(t);                          // save new tempo in project settings
  ssClockHelper_updateTempoDisplay(t);             // display tempo
}
// **********************************************
// *
// *    COUNT IN
// *
// **********************************************
void ssClockHelper_startCountIn(){
  _inCountIn = 33; // 32 step count in
  ssClockHelper_setCountInLED(ON);
  ssClockHelper_start();
}
void ssClockHelper_clearCountIn(){
  if (_inCountIn > 0) {
    ssPanel.setClick(OFF);                                  // turn off metro
    ssClockHelper_setCountInLED(OFF);  
    _inCountIn = 0;
  }
}

// **********************************************
// *
// *    SSClock Event Handlers
// *
// **********************************************
void ssClockHelper_handleClockStarted(){

}
void ssClockHelper_handleClockStopped(){
}
void ssClockHelper_handleClock32Note(bool outPhase){
  if ((_inCountIn > 0) && (_inCountIn % 16)) { ssPanel.setClick(OFF); }    // turns metro buzz off; downbeat stays on longer
  if ((_inRec && _inPlay && _metro) && (_currentStep % 16)) { ssPanel.setClick(OFF); }    // turns metro buzz off; downbeat stays on longer

  if (outPhase && _inPlay && _currentStep > -1) { 
    if (!_inCountIn){
      ssNoteData.playbackStep(_currentBank, _currentStep*2 + 1);            // the sequencer runs at 256 steps (1/16 notes), but recorder has 1/32 steps, this plays the outOfPhase steps
      ssArp_playbackStep(_currentBank, _currentStep, true);                 // the sequencer runs at 256 steps (1/16 notes), but arp has 1/32 steps noteOffs for stacato 1/16 notes, this plays the outOfPhase steps
      ssTuring_playbackStep(_currentBank, _currentStep, true);
      ssChord_playbackStep(_currentBank, _currentStep, true);
    }
  }
  if (ssNoteData.DEBUG_getEventQueueCount() > 0){  Serial.print("EQC:");Serial.println(ssNoteData.DEBUG_getEventQueueCount());  }
}
void ssClockHelper_handleClock16Note(bool outPhase){
  if (_inPlay){
    ssClockHelper_advanceNextSequencerStep();
    ssPanel.togglePort(0);      // TODO: make this a optional clock out setting
    
    if (_inRatchet) { 
      _currentStep--; 
      if (_currentStep < 0) { _currentStep == MAXSTEPS - 1; }
      ssMidiHelper_killAllOutNotes(SEQ);
    }
    ssDisplayUI_updateStepDisplay();
    
    if (_currentStep == 0 && _currentBank != _nextBank) {
      ssSequence_changeBank(_nextBank);
    }
    
    if (_inErase){
      if (_inTrigger){
        ssTrigData.eraseAtStep(_currentBank, _currentTriggerTrack, _currentStep);
      } else {
        ssMidiHelper_killTrackOutNotes(_currentNoteTrack, SEQ);
        ssNoteData.eraseAtStep(_currentBank, _currentNoteTrack, _currentStep*2);
      }
      ssFileHelper_makeFileDirty(47);    
    }

    ssTrigData.playbackStep(_currentBank, _currentStep); 
    //Serial.print("playBackStep::");Serial.print(_currentStep*2);Serial.println("-->"); 
    ssNoteData.playbackStep(_currentBank, _currentStep*2);
    ssArp_playbackStep(_currentBank, _currentStep, false);
    ssTuring_playbackStep(_currentBank, _currentStep, false);
    ssChord_playbackStep(_currentBank, _currentStep, false);

    if (_inFill && _inTrigger){
      if (_inFill == 2){
        ssSequencer_handleTriggerHit(true);
      } else {
        ssSequencer_handleTriggerHit(false);
      }
    }
  } else if (_inCountIn > 0) {
    if ((_inCountIn) % 4 == 1) { ssPanel.setClick(ON); }                                  // turns metro buzz for count in on every quarter note
    if (_inCountIn == 2) {
      ssClockHelper_setRecordTransport(ON);
      ssClockHelper_setPlayTransport(ON);
    }
    _inCountIn--;  
  }
  if ((_inRec && _inPlay && _metro) && _currentStep % 4 == 0) { ssPanel.setClick(ON); }   // turns metro buzz on every quarter note
  ssClock.clearOffset();
}
void ssClockHelper_handleClock8Note(bool outPhase){
}
void ssClockHelper_handleClock4Note(bool outPhase){
}

// **********************************************
// *
// *    SSCLOCK WRAPPER FUNCTIONS
// *
// **********************************************
void ssClockHelper_begin(int tempo){
  // set CLOCK handlers
  ssClock.setHandlerStarted(ssClockHelper_handleClockStarted);
  ssClock.setHandlerStopped(ssClockHelper_handleClockStopped);
  ssClock.setHandler32Note(ssClockHelper_handleClock32Note);
  ssClock.setHandler16Note(ssClockHelper_handleClock16Note);
  ssClock.setHandler8Note(ssClockHelper_handleClock8Note);
  ssClock.setHandler4Note(ssClockHelper_handleClock4Note);
  ssClock.begin(tempo);
  ssClockHelper_setClockSource(ssSettings_getGlobalClockSource());
  ssClockHelper_setTempo(tempo); 
}
void ssClockHelper_advanceClock(){          // TODO - add clock source; external midi in and external trigger in
  ssClock.advanceClock();
}
void ssClockHelper_setClockSource(byte s){
  ssClock.setClockSource(s);
  ssClockHelper_attachExtClockInterupt(s > 1);          // if clock source is external, turn on interupt
}
void ssClockHelper_stop(){
  ssClock.stop();
}
void ssClockHelper_start(){
  ssClock.start();
}
int ssClockHelper_getOffset(){
  return ssClock.getOffset();
}
void ssClockHelper_updateSwingSettings(){
  //Serial.println("ssClockHelper_updateSwingSettings");
  ssClock.swingEnable(ssSettings_getSwingEnable());
  ssClock.setSwing(ssSettings_getSwing());
  ssClock.swingType(ssSettings_getSwingType());
}

// **********************************************
// *
// *    TRANSPORT CONTROL STATE DISPLAY
// *
// **********************************************

// Play
void ssClockHelper_setPlayLED(bool inPlay){
  ssPanel.clearPoint(ROW_BANKAUX,GRNBTNLED);
  if (inPlay) { ssPanel.setPoint(ROW_BANKAUX,SOLID,GRNBTNLED,ON); }
}
// Record
void ssClockHelper_setRecLED(bool b){
  ssPanel.clearPoint(ROW_BANKAUX,REDBTNLED); 
  if (b){
    if (_recUnQuantized){
      ssPanel.setPoint(ROW_BANKAUX,DOUBLEBLINK,REDBTNLED,ON); 
    } else {
      ssPanel.setPoint(ROW_BANKAUX,SOLID,REDBTNLED,ON); 
    }
  }
}

// Stop - there is no display for STEP

// Count In
void ssClockHelper_setCountInLED(bool on){
  ssPanel.clearPoint(ROW_BANKAUX,REDBTNLED); 
  ssPanel.clearPoint(ROW_BANKAUX,GRNBTNLED);
  if (on){
    ssPanel.setPoint(ROW_BANKAUX,BLINK,GRNBTNLED,ON);
    ssPanel.setPoint(ROW_BANKAUX,BLINK,REDBTNLED,ON);
  } 
}

// TEMPO
void ssClockHelper_updateTempoDisplay(int t){
  ssClockHelper_updateTempoDisplay(t, false);
}

int mDisplayedTempo;
void ssClockHelper_updateTempoDisplay(int t, bool force){
  if (_panelPage == DEFAULT) {
    if (ssSettings_getGlobalClockSource() == CLOCKTYPE_MIDI) {
        ssPanel.display.setCursor(6,0);
        ssPanel.display.print(str_MIDI); 
    } else {
      if (force || (t != mDisplayedTempo)){    // only update the display if the integer tempo changed
        ssPanel.display.setCursor(6,0);
        char tmp[9];
        itoa(t,tmp,10);
        strcat(tmp,"     ");
        ssPanel.display.print(tmp); 
        mDisplayedTempo = t;
      }
    }
  }
}

// **********************************************
// *
// *    TRANSPORT CONTROL UI HANDLERS
// *
// **********************************************

void ssClockHelper_handlePlayButton(bool inShift){
  if (inShift) {
    if (!_inPlay){ 
      ssClockHelper_startCountIn();                         // if shift+Play start count in
    } else {
      ssClockHelper_clearCountIn();                         // if PLAY is pressed during count in then cancel count in
    }
  } else {
    ssClockHelper_clearCountIn();                         // if PLAY is pressed during count in then cancel count in
    ssClockHelper_togglePlayTransport();                           // handle play button press
  }  
}
void ssClockHelper_handleRecordButton(bool inShift){
  if (inShift && !_inPlay){                               // if shift+rec, toggle step record
    if (_inStepRec){
      ssSequencer_exitStepRecord();
    } else {
      ssSequencer_enterStepRecord();        
    }
  } else {
    if (_inStepRec) {
      ssSequencer_exitStepRecord();
    } else {
      ssClockHelper_toggleRecordTransport();                         // else toggle record
    }
  }  
}
void ssClockHelper_handleStopButton(bool inShift){
  if (_inCountIn) { ssClockHelper_clearCountIn();  }      // stop countIn
  if (_inStepRec) { 
    ssSequencer_exitStepRecord();
  }      // if in stepRec exit it
  ssClockHelper_stopTransport();                          // stop the transport; if we are already stopped this will just reset things like currentStep

  ssFileHelper_Backup();
  
  ssNoteData.DEBUG_printEventData(DEBUG_SHOWSTEPS);                    // do some helpful debug stuff
  ssNoteData.DEBUG_printNumOfEvents();
  //ssTrigData.DEBUG_printEventData(_currentBank);  
  ssSettings_DEBUGPrintGlobalSettings();
}
void ssClockHelper_handleKnobTurn(int delta){
  if (ssSettings_getGlobalClockSource() == 0) {               // only change the tempo if using internal clock
    ssClockHelper_setTempo(ssSettings_getTempo() + delta);  
  }
}

/*
 * 
 * HANDLE EXTERNAL CLOCK INTERUPT
 * 
 * 
 */
void ssClockHelper_attachExtClockInterupt(bool b){
  if (b) {
    // external clock interupt
    attachInterrupt(digitalPinToInterrupt(CLOCKIN), ssClockHelper_handleExtClockIn, RISING);  // external clock is interupt driven
  } else {
    detachInterrupt(digitalPinToInterrupt(CLOCKIN));
  }
}
 
byte mDisplayCount = 0;
void ssClockHelper_handleExtClockIn(){
  if (ssSettings_getGlobalClockSource() < 2) { return; }      // not using external clock
  float newTempo = ssClock.sync();
  ssSettings_setTempo(newTempo);
  if (mDisplayCount % 5 == 0 && !_inPopup) { ssClockHelper_updateTempoDisplay(newTempo); }
  mDisplayCount++;
}

void ssClockHelper_handleMidiClockIn(){
  if (ssSettings_getGlobalClockSource() != 1){ return; }
  ssClock.advanceClockTick();
  ssClock.advanceClockTick();
}
void ssClockHelper_handleMidiStart(){
  if (ssSettings_getGlobalClockSource() != 1){ return; }
  ssClockHelper_stopTransport(false);
  ssClockHelper_setPlayTransport(true,false);
}
void ssClockHelper_handleMidiContinue(){
  if (ssSettings_getGlobalClockSource() != 1){ return; }
  ssClockHelper_setPlayTransport(true,false);
}
void ssClockHelper_handleMidiStop(){
  if (ssSettings_getGlobalClockSource() != 1){ return; }
  ssClockHelper_setPlayTransport(false,false);
}
  
